using System.Linq;
using gov.va.med.vbecs.Common;
using System.Collections.Generic;
using System.Data;
using gov.va.med.vbecs.Common.Extensions;
using System;
using gov.va.med.vbecs.DAL.HL7AL;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using System.Collections;

namespace gov.va.med.vbecs.BOL
{
    /// <summary>
    /// PendingSpecimenTest, used to get pending blood units for user review
    /// </summary>
    public class PendingSpecimenTest
    {
        /// <summary>
        /// Determine if test results exist for a given divisiona and PendingTestStatus
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestStatus"></param>
        /// <returns>true if tests exist, otherwise false</returns>
        public static bool DoTestsExist(string divisionCode, PendingTestStatus pendingTestStatus)
        {
            if (string.IsNullOrWhiteSpace(divisionCode)) throw new ArgumentNullException("divisionCode");

            return DAL.PendingSpecimenTest.GetCount(divisionCode, pendingTestStatus) > 0;
        }

        /// <summary>
        /// Call data access layer to return data table of blood unit test results for given blood unit and test type id.
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestStatus"></param>
        /// <returns></returns>
        public static IList<PatientWithPendingSpecimenTestModel> GetPatientsWithPendingSpecimenTests(string divisionCode, PendingTestStatus pendingTestStatus)
        {
            if (string.IsNullOrWhiteSpace(divisionCode)) throw new ArgumentNullException("divisionCode");

            var patientList = new List<PatientWithPendingSpecimenTestModel>();

            // Get PendingSpecimenTestResults from the database
            var dsAllPatientsWithSpecimenTests = DAL.PendingSpecimenTest.GetPatientsWithPendingSpecimenTests(divisionCode, pendingTestStatus);
            // Convert DataTable to list of PatientWithSpecimenDao
            var pendingPatientTestDaoList = dsAllPatientsWithSpecimenTests.Tables[0].ToList<PatientWithSpecimenDao>();

            // Convert DataTable to list of PatientWithSpecimenDao
            var pendingOrderableTestDaoList = dsAllPatientsWithSpecimenTests.Tables[1].ToList<PatientSpecimenOrderableDao>();

            var patientDetailsDictionary = new Dictionary<Guid, Patient>();


            // Loop through each set of results and build the PendingBloolUnitAboRhModel and PendingSpecimenAntigenTypingModel
            foreach (var patientDao in pendingPatientTestDaoList)
            {
                var patientModel = new PatientWithPendingSpecimenTestModel();
                patientModel.SpecimenUid = patientDao.SpecimenUid;
                patientModel.PatientSpecimenGuid = patientDao.PatientSpecimenGuid;

                // Fetching a full patient details can be expensive so cache patient data
                // instead of fetching for every specimen
                Patient patientDetails;
                if (patientDetailsDictionary.TryGetValue(patientDao.PatientGuid, out patientDetails))
                {
                    patientModel.Patient = patientDetails;
                }
                else
                {
                    patientModel.Patient = new Patient(patientDao.PatientGuid);
                    patientDetailsDictionary.Add(patientDao.PatientGuid, patientModel.Patient);
                }

                patientModel.OrderableTestList = pendingOrderableTestDaoList.Where(x => x.PatientSpecimenGuid == patientDao.PatientSpecimenGuid)
                                                                            .Select(s => new OrderableTestModel(s.OrderableTestId, s.OrderableTestName)).ToList();
                patientList.Add(patientModel);
            }

            return patientList;
        }

        /// <summary>
        /// Call data access layer to return a list of specimen test results for given specimenUid, division code and test type id.
        /// </summary>
        /// <param name="patient"></param>
        /// <param name="patientSpecimenGuid"></param>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestStatus"></param>
        /// <param name="orderableTest"></param>
        /// <returns></returns>
        public static IList<IPendingSpecimenTestModel> GetPendingSpecimenTests(Patient patient, Guid patientSpecimenGuid, string divisionCode, PendingTestStatus pendingTestStatus, OrderableTest? orderableTest)
        {
            if (patient == null) throw new ArgumentNullException("patient");
            if (string.IsNullOrWhiteSpace(divisionCode)) throw new ArgumentNullException("divisionCode");

            var pendingTestList = new List<IPendingSpecimenTestModel>();

            // Get PendingSpecimenTest data from the database
            var dsAllPendingSpecimenTests = DAL.PendingSpecimenTest.GetPendingSpecimenTests(patient.PatientGuid, patientSpecimenGuid, divisionCode, pendingTestStatus, orderableTest);

            // Convert DataTable to list of PendingSpecimenTestDao
            var pendingSpecimenDaoList = dsAllPendingSpecimenTests.Tables[0].ToList<PendingSpecimenTestDao>();

            // Group test records by the ResultSetGuid
            var resultSetList = pendingSpecimenDaoList.GroupBy(g => g.SpecimenResultSetGuid);

            // Loop through each set of results and build the PendingSpecimen models
            foreach (var resultSet in resultSetList)
            {
                var firstTestDao = resultSet.First();
                var resultOrderableTest = (OrderableTest)firstTestDao.OrderableTestId;

                if (PendingTestHelper.IsAntigenTypingOrderableTest(resultOrderableTest))
                {
                    pendingTestList.Add(BuildPendingSpecimenModel(new PendingSpecimenAntigenTypingModel(), patient, resultSet.ToList()));
                }
                else
                {
                    switch (resultOrderableTest)
                    {
                        case OrderableTest.TAS:
                            pendingTestList.Add(BuildPendingSpecimenTasModel(patient, resultSet.ToList()));
                            break;
                        case OrderableTest.ABORh:
                        case OrderableTest.RepeatABORh:
                            pendingTestList.Add(BuildPendingSpecimenModel(new PendingSpecimenAboRhModel(), patient, resultSet.ToList()));
                            break;
                        case OrderableTest.ABS:
                        case OrderableTest.RepeatABS:
                            pendingTestList.Add(BuildPendingSpecimenModel(new PendingSpecimenAbsModel(), patient, resultSet.ToList()));
                            break;
                        case OrderableTest.DAT:
                        case OrderableTest.DATComp:
                        case OrderableTest.DATIgG:
                        case OrderableTest.RepeatDAT:
                        case OrderableTest.RepeatDATComp:
                        case OrderableTest.RepeatDATIgG:
                            pendingTestList.Add(BuildPendingSpecimenModel(new PendingSpecimenDatModel(), patient, resultSet.ToList()));
                            break;
                        //case OrderableTest.XM:
                        //    pendingTestList.Add(BuildPendingSpecimenXMatchModel(patient, resultSet.ToList()));
                        //    break;
                    }
                }
            }

            pendingTestList.AddRange(GetPendingXMatchTests(patient, patientSpecimenGuid, divisionCode, pendingTestStatus));

            return pendingTestList;
        }

        private static IList<IPendingSpecimenTestModel> GetPendingXMatchTests(Patient patient, Guid patientSpecimenGuid, string divisionCode, PendingTestStatus pendingTestStatus)
        {
            var pendingTestList = new List<IPendingSpecimenTestModel>();
            var dsAllPendingSpecimenXMTests = DAL.PendingBloodUnitTest.GetPendingBloodUnitXMTests(patientSpecimenGuid, divisionCode, pendingTestStatus);

            var pendingSpecimenXmDaoList = dsAllPendingSpecimenXMTests.Tables[0].ToList<PendingSpecimenXmTestDao>();
            var pendingBloodTestXmDaoList = dsAllPendingSpecimenXMTests.Tables[1].ToList<PendingBloodUnitTestDao>();
            var bloodUnitXmDaoList = dsAllPendingSpecimenXMTests.Tables[2].ToList<BloodUnitDao>();

            // Group test records by the ResultSetGuid
            var resultSetList = pendingBloodTestXmDaoList.GroupBy(g => g.UnitResultSetGuid);

            // Loop through each set of results and build the PendingSpecimenXmModel
            foreach (var resultSet in resultSetList)
            {
                // ISInterp and AHG Interp don't linke to an orderable test id so we don't
                // want to select it as the first
                var firstPendingBloodUnitDao = resultSet.First(x => x.OrderableTestId > 0);
                var resultOrderableTest = (OrderableTest)firstPendingBloodUnitDao.OrderableTestId;

                if (resultOrderableTest != OrderableTest.XM &&
                    resultOrderableTest != OrderableTest.RepeatXM)
                {
                    // We only want to process XM tests. Should never really reach this code
                    // its just a safe double check
                    continue;
                }

                var xmModel = new PendingSpecimenXMatchModel();

                var specimenXmDao = pendingSpecimenXmDaoList.FirstOrDefault(x => x.OrderedUnitGuid == firstPendingBloodUnitDao.OrderedUnitGuid);
                var pendingBloodUnitDao = bloodUnitXmDaoList.Where(x => x.BloodUnitGuid == firstPendingBloodUnitDao.BloodUnitGuid).FirstOrDefault();

                if (specimenXmDao == null)
                {
                    throw new Exception("Unable to find matching specimen XM for BloodUnitGuid: " + firstPendingBloodUnitDao.OrderedUnitGuid.ToString());
                }

                // Assign Base/Common Data
                PendingTestHelper.AssignPropertiesFromDao(xmModel, resultSet);

                // Assign IPendingSpecimenTestModel specific properties
                xmModel.PatientSpecimenGuid = specimenXmDao.PatientSpecimenGuid;
                xmModel.SpecimenUid = specimenXmDao.SpecimenUid;
                xmModel.SpecimenExpirationDate = specimenXmDao.SpecimenExpirationDate;
                xmModel.PatientGuid = specimenXmDao.PatientGuid;
                xmModel.CprsOrderNumber = specimenXmDao.OrderedTestCprsOrderNumber;
                xmModel.OrderedComponentCprsOrderNumber = specimenXmDao.OrderedComponentCprsOrderNumber;

                if (specimenXmDao.OrderedTestGuid.HasValue &&
                    specimenXmDao.OrderedTestPatientOrderGuid.HasValue)
                {
                    xmModel.OrderedTest = new OrderedTest(specimenXmDao.OrderedTestGuid.Value, specimenXmDao.OrderedTestPatientOrderGuid.Value);
                    xmModel.OrderedTest.OrderableTest = xmModel.OrderableTest.Name; // Name doesn't seem to fill so we will set it
                    xmModel.OrderedTest.IsReadOnly = false;
                    xmModel.OrderedTest.Patient = patient;
                }

                // Assign IPendingBloodUnitTestModel specific properties
                if (specimenXmDao.OrderedUnitGuid.HasValue)
                {
                    // Defect 284963
                    xmModel.OrderedUnitModel = new OrderedUnitModel(specimenXmDao.OrderedUnitGuid.Value, specimenXmDao.OrderedUnitSelectedDate, specimenXmDao.OrderedUnitRowVersion);
                }

                xmModel.OrderedTestGuid = firstPendingBloodUnitDao.OrderedTestGuid;
                xmModel.OrderedComponentGuid = firstPendingBloodUnitDao.OrderedComponentGuid;
                xmModel.BloodUnitId = firstPendingBloodUnitDao.BloodUnitId;
                xmModel.ResultSetGuid = firstPendingBloodUnitDao.UnitResultSetGuid;
                xmModel.BloodUnitModel = PendingTestHelper.GetBloodUnitModel(pendingBloodUnitDao);
                xmModel.TestValidation = xmModel.Validate(xmModel.BloodUnitModel);

                pendingTestList.Add(xmModel);
            }

            return pendingTestList;
        }

        private static IPendingSpecimenTestModel BuildPendingSpecimenTasModel(Patient patient, IList<PendingSpecimenTestDao> pendingTestDaoList)
        {
            var tasModel = new PendingSpecimenTasModel();

            // Assign Base/Common Data
            PendingTestHelper.AssignPropertiesFromDao(tasModel, pendingTestDaoList);

            // Assign Pending Specimen specific properties
            AssignPendingSpecimenProperties(tasModel, patient, pendingTestDaoList);

            var absBloodTestTypeList = PendingSpecimenAbsModel.GetTasBloodTestTypeIds();

            // Build the Abo/Rh Model            
            var aboRhTestList = pendingTestDaoList.Where(x => !absBloodTestTypeList.Contains((TestType)x.BloodTestTypeId)).ToList();
            if (aboRhTestList.Any())
            {
                tasModel.PendingSpecimenAboRhModel = new PendingSpecimenAboRhModel();
                BuildPendingSpecimenModel(tasModel.PendingSpecimenAboRhModel, patient, aboRhTestList);
            }

            // Build the Abs Model
            var absTestList = pendingTestDaoList.Where(x => absBloodTestTypeList.Contains((TestType)x.BloodTestTypeId)).ToList();
            if (absTestList.Any())
            {
                tasModel.PendingSpecimenAbsModel = new PendingSpecimenAbsModel();
                BuildPendingSpecimenModel(tasModel.PendingSpecimenAbsModel, patient, absTestList);
            }

            return tasModel;
        }

        private static IPendingSpecimenTestModel BuildPendingSpecimenModel(IPendingSpecimenTestModel pendingSpecimenTestModel, Patient patient, IList<PendingSpecimenTestDao> pendingTestDaoList)
        {
            // Assign Base/Common Data
            PendingTestHelper.AssignPropertiesFromDao(pendingSpecimenTestModel, pendingTestDaoList);

            // Assign Pending Specimen specific properties
            AssignPendingSpecimenProperties(pendingSpecimenTestModel, patient, pendingTestDaoList);

            return pendingSpecimenTestModel;
        }

        private static void AssignPendingSpecimenProperties(IPendingSpecimenTestModel pendingSpecimenTestModel, Patient patient, IList<PendingSpecimenTestDao> pendingTestDaoList)
        {
            var firstDao = pendingTestDaoList.First();

            pendingSpecimenTestModel.PatientSpecimenGuid = firstDao.PatientSpecimenGuid;
            pendingSpecimenTestModel.SpecimenUid = firstDao.SpecimenUid;
            pendingSpecimenTestModel.SpecimenExpirationDate = firstDao.SpecimenExpirationDate;
            pendingSpecimenTestModel.ResultSetGuid = firstDao.SpecimenResultSetGuid;
            pendingSpecimenTestModel.OrderedTest = new OrderedTest(firstDao.OrderedTestGuid, firstDao.PatientOrderGuid);
            pendingSpecimenTestModel.OrderedTest.OrderableTest = pendingSpecimenTestModel.OrderableTest.Name; // Name doesn't seem to fill so we will set it
            pendingSpecimenTestModel.OrderedTest.IsReadOnly = false;
            pendingSpecimenTestModel.OrderedTest.Patient = patient;
            pendingSpecimenTestModel.CprsOrderNumber = firstDao.CprsOrderNumber;
            pendingSpecimenTestModel.TestValidation = pendingSpecimenTestModel.Validate();
        }

        /// <summary>
        /// RejectTests
        /// BR_115.05
        /// </summary>
        /// <param name="pendingTestList"></param>
        public static bool RejectTests(IList<IPendingSpecimenTestModel> pendingTestList)
        {
            if (pendingTestList == null || !pendingTestList.Any()) return false;

            if (pendingTestList.Any(x => x is PendingSpecimenXMatchModel))
            {
                return RejectXMTests(pendingTestList.Select(x => x as PendingSpecimenXMatchModel).ToList());
            }

            ArrayList allTables = new ArrayList();
            ArrayList allSprocs = new ArrayList();

            // Buid DataTable to update/reject Pending Blood Unit Tests
            DataTable dtPendingTestReject = GetEmptyPendingSpecimenTestTableSchema();
            DataTable dtOrderedTestStatusUpdateToPrevious = GetEmptyOrderedTestTableSchema();
            DataTable dtOrderedTestStatusUpdate = GetEmptyOrderedTestTableSchemaForStatusUpdate();

            foreach (var pendingSpecimen in pendingTestList)
            {
                if (string.IsNullOrWhiteSpace(pendingSpecimen.ReviewComment))
                {
                    throw new Exception(StrRes.ValidMsg.UC115.ReviewCommentRequired().ResString);
                }

                foreach (var testResult in pendingSpecimen.TestResultList)
                {
                    DataRow dr = dtPendingTestReject.NewRow();
                    dr[TABLE.PendingSpecimenTest.PendingSpecimenTestId] = testResult.PendingTestId;
                    dr[TABLE.PendingSpecimenTest.SpecimenTestGuid] = DBNull.Value;
                    dr[TABLE.PendingSpecimenTest.PendingTestStatusId] = (byte)PendingTestStatus.Rejected;
                    dr[TABLE.PendingSpecimenTest.RejectionComment] = pendingSpecimen.ReviewComment;
                    dr[TABLE.PendingSpecimenTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    dr[TABLE.PendingSpecimenTest.RowVersion] = testResult.RowVersion;
                    dr[TABLE.PendingSpecimenTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115SpecimenTabView;
                    dtPendingTestReject.Rows.Add(dr);
                }

                // Update Order and Task Statuses
                bool revertToPreviousStatus = true;
                if (pendingSpecimen.OrderableTest.OrderableTestId == OrderableTest.TAS)
                {
                    TaskStatus origTaskStatus = Utility.GetTaskStatusCodeFromString(pendingSpecimen.OrderedTest.TaskStatusCode.ToString());
                    OrderStatus origOrderStatus = pendingSpecimen.OrderedTest.OrderStatusCode;
                    TaskStatus newTaskStatus;
                    OrderStatus newOrderStatus;
                    GetTaskStatusForTasRejecting(pendingSpecimen, pendingTestList, out newTaskStatus, out newOrderStatus, ref revertToPreviousStatus);

                    // Only update if we have changes and we don't already have an update to the OrderedTest table for this OrderedTestGuid
                    if (!revertToPreviousStatus &&
                        dtOrderedTestStatusUpdate.Select(TABLE.OrderedTest.OrderedTestGuid + " = '" + pendingSpecimen.OrderedTest.OrderedTestGuid.ToString() + "'").Length == 0 &&
                        (newTaskStatus != origTaskStatus ||
                        newOrderStatus != origOrderStatus))
                    {
                        var dr = dtOrderedTestStatusUpdate.NewRow();
                        dr[TABLE.OrderedTest.OrderedTestGuid] = pendingSpecimen.OrderedTest.OrderedTestGuid;
                        dr[TABLE.OrderedTest.TaskStatusCode] = Utility.GetTaskStatusCodeFromEnum(newTaskStatus);
                        dr[TABLE.OrderedTest.OrderStatusCode] = Utility.GetOrderStatusCodeFromEnum(newOrderStatus);
                        dr[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                        dr[TABLE.OrderedTest.LastUpdateFunctionId] = UpdateFunction.UC115SpecimenTabView;
                        dr[TABLE.OrderedTest.RowVersion] = pendingSpecimen.OrderedTest.RowVersion;
                        dtOrderedTestStatusUpdate.Rows.Add(dr);
                    }
                }

                if (revertToPreviousStatus &&
                    dtOrderedTestStatusUpdateToPrevious.Select(TABLE.OrderedTest.OrderedTestGuid + " = '" + pendingSpecimen.OrderedTest.OrderedTestGuid.ToString() + "'" ).Length == 0)
                {
                    // The Diagnostic test returns to the previous status and ready for testing.
                    DataRow drOrderedTestStatus = dtOrderedTestStatusUpdateToPrevious.NewRow();
                    drOrderedTestStatus[TABLE.OrderedTest.OrderedTestGuid] = pendingSpecimen.OrderedTest.OrderedTestGuid;
                    drOrderedTestStatus[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    drOrderedTestStatus[TABLE.OrderedTest.RowVersion] = pendingSpecimen.OrderedTest.RowVersion;
                    drOrderedTestStatus[TABLE.OrderedTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115SpecimenTabView;
                    dtOrderedTestStatusUpdateToPrevious.Rows.Add(drOrderedTestStatus);
                }
            }

            allTables.Add(dtPendingTestReject);
            allSprocs.Add(STOREDPROC.UpdatePendingSpecimenTest.StoredProcName);

            if (dtOrderedTestStatusUpdateToPrevious != null &&
                dtOrderedTestStatusUpdateToPrevious.Rows.Count > 0)
            {
                allTables.Add(dtOrderedTestStatusUpdateToPrevious);
                allSprocs.Add(STOREDPROC.UpdateOrderedTestToPreviousTaskStatus.StoredProcName);
            }

            if (dtOrderedTestStatusUpdate != null &&
                dtOrderedTestStatusUpdate.Rows.Count > 0)
            {
                allTables.Add(dtOrderedTestStatusUpdate);
                allSprocs.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
            }

            // Execute Transactions
            int retVal = new Common.StoredProcedure().TransactionalGetValue(allSprocs, allTables);
            return (retVal == 0);
        }

        /// <summary>
        /// Reject XMatch Tests
        /// </summary>
        /// <param name="pendingTestList"></param>
        /// <returns></returns>
        private static bool RejectXMTests(IList<PendingSpecimenXMatchModel> pendingTestList)
        {
            if (pendingTestList == null || !pendingTestList.Any()) return false;

            ArrayList allTables = new ArrayList();
            ArrayList allSprocs = new ArrayList();

            // Buid DataTable to update/reject Pending Blood Unit Tests
            DataTable dtPendingTestReject = DAL.PendingBloodUnitTest.GetEmptyPendingBloodUnitTestTableSchemaForUpdate();
            DataTable dtOrderedTestStatusUpdateToPrevious = GetEmptyOrderedTestTableSchema();

            foreach (var pendingTest in pendingTestList)
            {
                if (string.IsNullOrWhiteSpace(pendingTest.ReviewComment))
                {
                    throw new Exception(StrRes.ValidMsg.UC115.ReviewCommentRequired().ResString);
                }

                foreach (var testResult in pendingTest.TestResultList)
                {
                    DataRow dr = dtPendingTestReject.NewRow();
                    dr[TABLE.PendingBloodUnitTest.PendingBloodUnitTestId] = testResult.PendingTestId;
                    dr[TABLE.PendingBloodUnitTest.BloodUnitTestGuid] = DBNull.Value;
                    dr[TABLE.PendingBloodUnitTest.PendingTestStatusId] = (byte)PendingTestStatus.Rejected;
                    dr[TABLE.PendingBloodUnitTest.RejectionComment] = pendingTest.ReviewComment;
                    dr[TABLE.PendingBloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    dr[TABLE.PendingBloodUnitTest.RowVersion] = testResult.RowVersion;
                    dr[TABLE.PendingBloodUnitTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115BloodUnitTabView;
                    dtPendingTestReject.Rows.Add(dr);
                }

                if (pendingTest.OrderedTest != null)
                {
                    // The Diagnostic test returns to the previous status and ready for testing.
                    DataRow drOrderedTestStatus = dtOrderedTestStatusUpdateToPrevious.NewRow();
                    drOrderedTestStatus[TABLE.OrderedTest.OrderedTestGuid] = pendingTest.OrderedTest.OrderedTestGuid;
                    drOrderedTestStatus[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    drOrderedTestStatus[TABLE.OrderedTest.RowVersion] = pendingTest.OrderedTest.RowVersion;
                    drOrderedTestStatus[TABLE.OrderedTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115SpecimenTabView;
                    dtOrderedTestStatusUpdateToPrevious.Rows.Add(drOrderedTestStatus);
                }
            }

            allTables.Add(dtPendingTestReject);
            allSprocs.Add(STOREDPROC.UpdatePendingBloodUnitTest.StoredProcName);

            if (dtOrderedTestStatusUpdateToPrevious != null &&
                dtOrderedTestStatusUpdateToPrevious.Rows.Count > 0)
            {
                allTables.Add(dtOrderedTestStatusUpdateToPrevious);
                allSprocs.Add(STOREDPROC.UpdateOrderedTestToPreviousTaskStatus.StoredProcName);
            }

            // Execute Transactions
            int retVal = new Common.StoredProcedure().TransactionalGetValue(allSprocs, allTables);
            return (retVal == 0);
        }

        /// <summary>
        /// Accept Tests
        /// BR_115.07
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestList"></param>
        /// <param name="isWorkLoadDefined"></param>
        /// <returns></returns>
        public static bool AcceptTests(string divisionCode, IList<IPendingSpecimenTestModel> pendingTestList, ref bool isWorkLoadDefined)
        {
            if (pendingTestList == null || !pendingTestList.Any()) return false;

            if (pendingTestList.Any(x => x is PendingSpecimenXMatchModel))
            {
                // Process XMatch Results
                return AcceptXMTests(divisionCode,
                                     pendingTestList.Select(x => x as PendingSpecimenXMatchModel).ToList(),
                                     ref isWorkLoadDefined);
            }

            ArrayList allTables = new ArrayList();
            ArrayList allSprocs = new ArrayList();

            // Add to SpecimenTests and Update PendingSpecimenTests
            DataTable dtSpecimenTests = null;
            DataTable dtPendingSpecimenTests = null;
            BuildSpecimenTestTablesForAccepting(divisionCode, pendingTestList, ref dtSpecimenTests, ref dtPendingSpecimenTests);

            allTables.Add(dtSpecimenTests);
            allSprocs.Add(STOREDPROC.InsertSpecimenTest.StoredProcName);

            allTables.Add(dtPendingSpecimenTests);
            allSprocs.Add(STOREDPROC.UpdatePendingSpecimenTest.StoredProcName);

            // Generate Exception Report for each ExceptionType
            // Append on Exception Reports
            var exceptionReportList = ExceptionReportCreator.CreateExceptionReports(pendingTestList);
            if (exceptionReportList != null &&
                exceptionReportList.Any())
            {
                foreach (var exception in exceptionReportList)
                {
                    DataTable dtException = exception.ExceptionData.Table;
                    dtException.Rows.Add(exception.ExceptionData.ItemArray);
                    allTables.Add(dtException);
                    allSprocs.Add(ExceptionReport.GetStoredProcName(exception.ExceptionType));
                }
            }

            // Append WorkLoads
            DataTable dtWorkLoad = null;
            PendingTestHelper.BuildWorkLoadTableForAccepting(pendingTestList, UpdateFunction.UC115SpecimenTabView, ref dtWorkLoad, ref isWorkLoadDefined);

            if (dtWorkLoad != null &&
                dtWorkLoad.Rows.Count > 0)
            {
                allTables.Add(dtWorkLoad);
                allSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
            }

            // Update Patient Task List 
            // BR_115.07
            var dtOrderedTestForVista = new DataTable();
            var dtOrderedTestTaskStatus = new DataTable();
            BuildOrderTaskStatusTableForAccepting(pendingTestList, ref dtOrderedTestTaskStatus, ref dtOrderedTestForVista);

            if (dtOrderedTestTaskStatus != null &&
                dtOrderedTestTaskStatus.Rows.Count > 0)
            {
                allTables.Add(dtOrderedTestTaskStatus);
                allSprocs.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
            }

            // Execute Transactions
            bool success = new Common.StoredProcedure().TransactionalGetValue(allSprocs, allTables) == 0;

            // Send HL7 message to CPRS if order is now complete
            // BR_56.05
            if (success &&
                dtOrderedTestForVista != null &&
                dtOrderedTestForVista.Rows.Count > 0)
            {
                new VbecsOmgMessage().PatientOrderUpdate(dtOrderedTestForVista);
            }

            return success;
        }

        /// <summary>
        /// Accept XMatch Tests
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestList"></param>
        /// <param name="isWorkLoadDefined"></param>
        /// <returns></returns>
        public static bool AcceptXMTests(string divisionCode, IList<PendingSpecimenXMatchModel> pendingTestList, ref bool isWorkLoadDefined)
        {
            if (pendingTestList == null || !pendingTestList.Any()) return false;

            ArrayList allTables = new ArrayList();
            ArrayList allSprocs = new ArrayList();

            // Add to SpecimenTests and Update PendingSpecimenTests
            DataTable dtBloodUnitTests = null;
            DataTable dtPendingBloodUnitTests = null;
            BuildBloodUnitTestTablesForAccepting(divisionCode, pendingTestList, ref dtBloodUnitTests, ref dtPendingBloodUnitTests);

            allTables.Add(dtBloodUnitTests);
            allSprocs.Add(STOREDPROC.InsertBloodUnitTest.StoredProcName);

            allTables.Add(dtPendingBloodUnitTests);
            allSprocs.Add(STOREDPROC.UpdatePendingBloodUnitTest.StoredProcName);

            DataTable dtOrderedUnit = null;
            BuildOrderedUnitTable(divisionCode, pendingTestList, ref dtOrderedUnit);

            if (dtOrderedUnit != null &&
                dtOrderedUnit.Rows.Count > 0)
            {
                allTables.Add(dtOrderedUnit);
                allSprocs.Add(STOREDPROC.UpdateOrderedUnit.StoredProcName);
            }

            // Generate Exception Report for each ExceptionType
            // Append on Exception Reports
            var exceptionReportList = ExceptionReportCreator.CreateExceptionReports(pendingTestList);
            if (exceptionReportList != null &&
                exceptionReportList.Any())
            {
                foreach (var exception in exceptionReportList)
                {
                    DataTable dtException = exception.ExceptionData.Table;
                    dtException.Rows.Add(exception.ExceptionData.ItemArray);
                    allTables.Add(dtException);
                    allSprocs.Add(ExceptionReport.GetStoredProcName(exception.ExceptionType));
                }
            }

            // Append WorkLoads
            DataTable dtWorkLoad = null;
            PendingTestHelper.BuildWorkLoadTableForAccepting(pendingTestList, UpdateFunction.UC115SpecimenTabView, ref dtWorkLoad, ref isWorkLoadDefined);

            if (dtWorkLoad != null &&
                dtWorkLoad.Rows.Count > 0)
            {
                allTables.Add(dtWorkLoad);
                allSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
            }

            // Update Patient Task List 
            // BR_115.07
            var dtOrderedTestForVista = new DataTable();
            var dtOrderedTestTaskStatus = new DataTable();
            BuildOrderTaskStatusTableForAccepting(pendingTestList, ref dtOrderedTestTaskStatus, ref dtOrderedTestForVista);

            if (dtOrderedTestTaskStatus != null &&
                dtOrderedTestTaskStatus.Rows.Count > 0)
            {
                allTables.Add(dtOrderedTestTaskStatus);
                allSprocs.Add(STOREDPROC.UpdateOrderedTestTaskStatus.StoredProcName);
            }

            // Execute Transactions
            bool success = new Common.StoredProcedure().TransactionalGetValue(allSprocs, allTables) == 0;

            if (success)
            {
                // BR_56.31 - Send Order Completion message to Vista
                var cprsOrderNumberGroupList = pendingTestList.Where(w => w.OrderableTest.OrderableTestId == OrderableTest.XM
                                                                    && !string.IsNullOrWhiteSpace(w.OrderedComponentCprsOrderNumber)
                                                                    && w.OrderedComponentGuid.HasValue)
                                                              .GroupBy(x => x.OrderedComponentCprsOrderNumber);

                foreach (var cprsOrderNumberGroup in cprsOrderNumberGroupList)
                {
                    var pendingTestXM = cprsOrderNumberGroup.First();
                    var componentOrderStatus = BOL.OrderedComponent.GetStatusIncludingUnitsEligibleForIssue(pendingTestXM.OrderedComponentGuid.Value);

                    if (componentOrderStatus == Common.TaskStatus.Filled)
                    {
                        BOL.OrderedComponent.SendComponentOrderCompletionMessageToVista(cprsOrderNumberGroup.Key);
                    }
                }
            }

            return success;
        }

        /// <summary>
        /// Build data in DataTables for BloodUnitTest, BloodUnitStatus, and PendingloodUnitTest
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestList"></param>
        /// <param name="dtBloodUnitTests"></param>
        /// <param name="dtPendingBloodUnitTests"></param>
        private static void BuildBloodUnitTestTablesForAccepting(string divisionCode, IEnumerable<PendingSpecimenXMatchModel> pendingTestList, ref DataTable dtBloodUnitTests, ref DataTable dtPendingBloodUnitTests)
        {
            dtBloodUnitTests = BOL.BloodUnitTest.GetBloodUnitTestTableForXM(false);
            dtPendingBloodUnitTests = DAL.PendingBloodUnitTest.GetEmptyPendingBloodUnitTestTableSchemaForUpdate();

            foreach (var pendingTest in pendingTestList)
            {
                if (pendingTest.BloodUnitModel == null)
                {
                    throw new Exception("ProductCode must be selected before saving");
                }
                if (pendingTest.TestValidation.TestValidationStatus != TestValidationStatus.Valid &&
                    pendingTest.TestValidation.TestValidationStatus != TestValidationStatus.Warning)
                {
                    throw new Exception("Cannot accept a test that is not in a Valid or Warning validation status.");
                }

                TaskStatus previousTaskStatus = TaskStatus.Unknown;
                if (pendingTest.OrderedTestGuid.HasValue)
                {
                    previousTaskStatus = DAL.OrderedTest.GetOrderedTestToPreviousTaskStatus(pendingTest.OrderedTestGuid.Value);
                }

                foreach (var testResult in pendingTest.TestResultList)
                {
                    // Skip check cells for non-weak D typing
                    if (!pendingTest.OrderableTest.IsWeakD &&
                        testResult.TestType == TestType.Anti_Dc_AHG)
                    {
                        continue;
                    }

                    // Add to the BloodUnitTestTable
                    var drBUT = dtBloodUnitTests.NewRow();
                    Guid bloodUnitTestGuid = Guid.NewGuid();
                    drBUT[TABLE.BloodUnitTest.BloodUnitTestGuid] = bloodUnitTestGuid;
                    drBUT[TABLE.BloodUnitTest.BloodUnitGuid] = pendingTest.BloodUnitModel.BloodUnitGuid;
                    drBUT[TABLE.BloodUnitTest.BloodTestTypeId] = testResult.TestType;
                    drBUT[TABLE.BloodUnitTest.TestResultId] = testResult.TestResultId;
                    drBUT[TABLE.BloodUnitTest.TestDate] = pendingTest.TestedDateTime;
                    drBUT[TABLE.BloodUnitTest.TestTechId] = pendingTest.InstrumentUserId;
                    drBUT[TABLE.BloodUnitTest.EntryMethodCode] = Convert.ToString(Common.TestEntryMethod.Manual);
                    // BR_56.27
                    drBUT[TABLE.BloodUnitTest.CorrectedResultIndicator] = (previousTaskStatus == TaskStatus.ResultsCorrected);
                    drBUT[TABLE.BloodUnitTest.OrderedUnitGuid] = pendingTest.OrderedUnitModel.OrderedUnitGuid;
                    drBUT[TABLE.BloodUnitTest.OrderedComponentGuid] = DBNull.Value;
                    drBUT[TABLE.BloodUnitTest.OrderedTestGuid] = DBNull.Value;
                    drBUT[TABLE.BloodUnitTest.ReactivityPhaseCode] = DBNull.Value;
                    drBUT[TABLE.BloodUnitTest.DivisionCode] = pendingTest.DivisionCode;
                    drBUT[TABLE.BloodUnitTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115BloodUnitTabView;
                    drBUT[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
                    drBUT[TABLE.BloodUnitTest.TestingMethodCode] = Common.TestingMethod.AutomatedInstrument;
                    drBUT[TABLE.BloodUnitTest.RackGuid] = DBNull.Value;
                    drBUT[TABLE.BloodUnitTest.AutoInstrumentName] = pendingTest.InstrumentName;
                    drBUT[TABLE.BloodUnitTest.TestComments] = string.Empty;

                    if (pendingTest is PendingSpecimenXMatchModel)
                    {
                        drBUT[TABLE.BloodUnitTest.ReactivityPhaseCode] = ((int)ReactivityPhase.IAT).ToString();
                    }
                    if (pendingTest.OrderedComponentGuid.HasValue)
                    {
                        drBUT[TABLE.BloodUnitTest.OrderedComponentGuid] = pendingTest.OrderedComponentGuid.Value;
                    }
                    if (pendingTest.OrderedTestGuid.HasValue)
                    {
                        drBUT[TABLE.BloodUnitTest.OrderedTestGuid] = pendingTest.OrderedTestGuid.Value;
                    }
                    if (testResult.IsInterp)
                    {
                        drBUT[TABLE.BloodUnitTest.TestComments] = pendingTest.ReviewComment;
                    }

                    dtBloodUnitTests.Rows.Add(drBUT);

                    // Link to the PendingBloodUnitTest Table and update it's status
                    var drPendingUnit = dtPendingBloodUnitTests.NewRow();
                    drPendingUnit[TABLE.PendingBloodUnitTest.PendingBloodUnitTestId] = testResult.PendingTestId;
                    drPendingUnit[TABLE.PendingBloodUnitTest.BloodUnitTestGuid] = bloodUnitTestGuid;
                    drPendingUnit[TABLE.PendingBloodUnitTest.PendingTestStatusId] = (byte)PendingTestStatus.Accepted;
                    drPendingUnit[TABLE.PendingBloodUnitTest.RejectionComment] = DBNull.Value;
                    drPendingUnit[TABLE.PendingBloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    drPendingUnit[TABLE.PendingBloodUnitTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115BloodUnitTabView;
                    drPendingUnit[TABLE.PendingBloodUnitTest.RowVersion] = testResult.RowVersion;
                    dtPendingBloodUnitTests.Rows.Add(drPendingUnit);
                }
            }
        }

        /// <summary>
        /// Build OrderedUnit table for db changes 
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestList"></param>
        /// <param name="dtOrderedUnit"></param>
        private static void BuildOrderedUnitTable(string divisionCode, IEnumerable<PendingSpecimenXMatchModel> pendingTestList, ref DataTable dtOrderedUnit)
        {
            dtOrderedUnit = new DataTable(TABLE.OrderedUnit.TableName);
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.OrderedUnitGuid, typeof(Guid));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.CrossmatchDate, typeof(DateTime));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.CrossmatchTechId, typeof(string));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.CrossmatchResultCode, typeof(char));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.ReleaseDate, typeof(DateTime));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.ReleaseTechId, typeof(string));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.RecordStatusCode, typeof(char));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.RowVersion, typeof(byte[]));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.DivisionCode, typeof(string));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.LastUpdateUser, typeof(string));
            dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.LastUpdateFunctionId, typeof(int));

            foreach (var pendingTest in pendingTestList)
            {
                DataRow drOrderedUnit = dtOrderedUnit.NewRow();
                drOrderedUnit[TABLE.OrderedUnit.OrderedUnitGuid] = pendingTest.OrderedUnitModel.OrderedUnitGuid;
                drOrderedUnit[TABLE.OrderedUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(RecordStatusCode.Active);
                drOrderedUnit[TABLE.OrderedUnit.RowVersion] = pendingTest.OrderedUnitModel.RowVersion;
                drOrderedUnit[TABLE.OrderedUnit.DivisionCode] = divisionCode;
                drOrderedUnit[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
                drOrderedUnit[TABLE.OrderedUnit.LastUpdateFunctionId] = (int)UpdateFunction.UC115SpecimenTabView;
                drOrderedUnit[TABLE.OrderedUnit.CrossmatchDate] = DBNull.Value;
                drOrderedUnit[TABLE.OrderedUnit.CrossmatchTechId] = DBNull.Value;
                drOrderedUnit[TABLE.OrderedUnit.CrossmatchResultCode] = DBNull.Value;
                drOrderedUnit[TABLE.OrderedUnit.ReleaseTechId] = DBNull.Value;
                drOrderedUnit[TABLE.OrderedUnit.ReleaseDate] = DBNull.Value;
                var divisionDateTime = DAL.VbecsDateTime.GetDivisionCurrentDateTime();

                if (pendingTest.XMInterp.TestResultId == PendingTestHelper.COMPATIBLE_SAFE_TO_TRANSFUSE_ID ||
                    pendingTest.XMInterp.TestResultId == PendingTestHelper.INCOMPATIBLE_ONLY_MEDICAL_DIRECTOR_ID)
                {
                    drOrderedUnit[TABLE.OrderedUnit.CrossmatchDate] = divisionDateTime;
                    drOrderedUnit[TABLE.OrderedUnit.CrossmatchTechId] = pendingTest.InstrumentUserId;
                    drOrderedUnit[TABLE.OrderedUnit.CrossmatchResultCode] = pendingTest.XMInterp.TestResultId;
                }
                else if (pendingTest.XMInterp.TestResultId == PendingTestHelper.COMPATIBLE_DONT_TRANSFUSE_ID ||
                         pendingTest.XMInterp.TestResultId == PendingTestHelper.INCOMPATIBLE)
                {
                    drOrderedUnit[TABLE.OrderedUnit.CrossmatchDate] = divisionDateTime;
                    drOrderedUnit[TABLE.OrderedUnit.CrossmatchTechId] = pendingTest.InstrumentUserId;
                    drOrderedUnit[TABLE.OrderedUnit.CrossmatchResultCode] = pendingTest.XMInterp.TestResultId;
                    drOrderedUnit[TABLE.OrderedUnit.ReleaseTechId] = pendingTest.InstrumentUserId;
                    drOrderedUnit[TABLE.OrderedUnit.ReleaseDate] = divisionDateTime;
                }

                dtOrderedUnit.Rows.Add(drOrderedUnit);
            }
        }

        /// <summary>
        /// Build data in DataTables for SpecimenTest, SpecimenStatus, and PendingloodUnitTest
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestList"></param>
        /// <param name="dtSpecimenTests"></param>
        /// <param name="dtPendingSpecimenTests"></param>
        private static void BuildSpecimenTestTablesForAccepting(string divisionCode, IEnumerable<IPendingSpecimenTestModel> pendingTestList, ref DataTable dtSpecimenTests, ref DataTable dtPendingSpecimenTests)
        {
            dtSpecimenTests = BOL.SpecimenTest.InitializeSpecimenTestTable(false);
            dtPendingSpecimenTests = GetEmptyPendingSpecimenTestTableSchema();

            foreach (var pendingTest in pendingTestList)
            {
                if (pendingTest.OrderedTest == null)
                {
                    throw new Exception("OrderedTest must exist before saving");
                }
                if (pendingTest.TestValidation.TestValidationStatus != TestValidationStatus.Valid &&
                    pendingTest.TestValidation.TestValidationStatus != TestValidationStatus.Warning)
                {
                    throw new Exception("Cannot accept a test that is not in a Valid or Warning validation status.");
                }

                TaskStatus previousTaskStatus = TaskStatus.Unknown;
                if (pendingTest.OrderedTest != null)
                {
                    previousTaskStatus = DAL.OrderedTest.GetOrderedTestToPreviousTaskStatus(pendingTest.OrderedTest.OrderedTestGuid);
                }

                foreach (var testResult in pendingTest.TestResultList)
                {
                    // Skip check cells for non-weak D typing
                    if (!pendingTest.OrderableTest.IsWeakD &&
                        testResult.TestType == TestType.Anti_Dc_AHG)
                    {
                        continue;
                    }

                    // Add to the SpecimenTestTable
                    var drST = dtSpecimenTests.NewRow();
                    Guid specimenTestGuid = Guid.NewGuid();

                    drST[TABLE.SpecimenTest.SpecimenTestGuid] = specimenTestGuid;
                    drST[TABLE.SpecimenTest.OrderedTestGuid] = pendingTest.OrderedTest.OrderedTestGuid;
                    drST[TABLE.SpecimenTest.PatientSpecimenGuid] = pendingTest.PatientSpecimenGuid;
                    drST[TABLE.SpecimenTest.TestingMethodCode] = Common.TestingMethod.AutomatedInstrument;
                    drST[TABLE.SpecimenTest.AntibodyTypeId] = DBNull.Value;
                    drST[TABLE.SpecimenTest.CannedCommentGuid] = DBNull.Value;
                    drST[TABLE.SpecimenTest.OffsiteIndicator] = DBNull.Value;
                    drST[TABLE.SpecimenTest.OffsiteLocation] = DBNull.Value;
                    drST[TABLE.SpecimenTest.OffsiteCost] = DBNull.Value;
                    // BR_56.27
                    drST[TABLE.SpecimenTest.CorrectedResultIndicator] = (previousTaskStatus == TaskStatus.ResultsCorrected);
                    drST[TABLE.SpecimenTest.AboRhChangeIndicator] = DBNull.Value;
                    drST[TABLE.SpecimenTest.RackGuid] = DBNull.Value;
                    drST[TABLE.SpecimenTest.AutoInstrumentQCIndicator] = DBNull.Value;
                    drST[TABLE.SpecimenTest.AutoInstrumentName] = pendingTest.InstrumentName;
                    drST[TABLE.SpecimenTest.ReactivityPhaseCode] = DBNull.Value;
                    drST[TABLE.SpecimenTest.TestTechId] = pendingTest.InstrumentUserId;
                    drST[TABLE.SpecimenTest.BloodTestTypeId] = (int)testResult.TestType;
                    drST[TABLE.SpecimenTest.TestResultId] = testResult.TestResultId;
                    drST[TABLE.SpecimenTest.TestDate] = pendingTest.TestedDateTime;
                    drST[TABLE.SpecimenTest.DivisionCode] = pendingTest.DivisionCode;
                    drST[TABLE.SpecimenTest.TestComments] = string.Empty;

                    if (testResult.IsInterp)
                    {
                        drST[TABLE.SpecimenTest.TestComments] = pendingTest.ReviewComment;
                    }
                    if (pendingTest is PendingSpecimenAbsModel)
                    {
                        drST[TABLE.SpecimenTest.ReactivityPhaseCode] = ((int)ReactivityPhase.IAT).ToString();
                    }

                    dtSpecimenTests.Rows.Add(drST);

                    // Link to the PendingSpecimenTest Table and update it's status
                    var drPendingUnit = dtPendingSpecimenTests.NewRow();
                    drPendingUnit[TABLE.PendingSpecimenTest.PendingSpecimenTestId] = testResult.PendingTestId;
                    drPendingUnit[TABLE.PendingSpecimenTest.SpecimenTestGuid] = specimenTestGuid;
                    drPendingUnit[TABLE.PendingSpecimenTest.PendingTestStatusId] = (byte)PendingTestStatus.Accepted;
                    drPendingUnit[TABLE.PendingSpecimenTest.RejectionComment] = DBNull.Value;
                    drPendingUnit[TABLE.PendingSpecimenTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    drPendingUnit[TABLE.PendingSpecimenTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115SpecimenTabView;
                    drPendingUnit[TABLE.PendingSpecimenTest.RowVersion] = testResult.RowVersion;
                    dtPendingSpecimenTests.Rows.Add(drPendingUnit);
                }
            }


            if (dtSpecimenTests != null)
            {
                Common.Utility.AppendLastUpdateInformation(dtSpecimenTests, UpdateFunction.UC115SpecimenTabView);
            }
        }

        /// <summary>
        /// BR_115.07
        /// </summary>
        /// <param name="pendingTestList"></param>
        /// <param name="dtOrderedTest"></param>
        /// <param name="dtOrderedTestForVista"></param>        
        /// <returns></returns>
        private static void BuildOrderTaskStatusTableForAccepting(IEnumerable<IPendingSpecimenTestModel> pendingTestList, ref DataTable dtOrderedTest, ref DataTable dtOrderedTestForVista)
        {
            dtOrderedTest = GetEmptyOrderedTestTableSchemaForStatusUpdate();
            var groupedPendingTestList = pendingTestList.Where(w => w.OrderedTest != null).GroupBy(x => x.OrderedTest.OrderedTestGuid);

            foreach (var pendingSpecimenKey in groupedPendingTestList)
            {
                var pendingSpecimen = pendingSpecimenKey.FirstOrDefault();

                TaskStatus origTaskStatus = Utility.GetTaskStatusCodeFromString(pendingSpecimen.OrderedTest.TaskStatusCode.ToString());
                OrderStatus origOrderStatus = pendingSpecimen.OrderedTest.OrderStatusCode;
                TaskStatus newTaskStatus;
                OrderStatus newOrderStatus;
                GetTaskStatusForAccepting(pendingSpecimen, pendingTestList, out newTaskStatus, out newOrderStatus);

                // Only update if we have changes
                if (newTaskStatus != origTaskStatus ||
                    newOrderStatus != origOrderStatus)
                {
                    var dr = dtOrderedTest.NewRow();
                    dr[TABLE.OrderedTest.OrderedTestGuid] = pendingSpecimen.OrderedTest.OrderedTestGuid;
                    dr[TABLE.OrderedTest.TaskStatusCode] = Utility.GetTaskStatusCodeFromEnum(newTaskStatus);
                    dr[TABLE.OrderedTest.OrderStatusCode] = Utility.GetOrderStatusCodeFromEnum(newOrderStatus);
                    dr[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    dr[TABLE.OrderedTest.LastUpdateFunctionId] = UpdateFunction.UC115SpecimenTabView;
                    dr[TABLE.OrderedTest.RowVersion] = pendingSpecimen.OrderedTest.RowVersion;
                    dtOrderedTest.Rows.Add(dr);

                    if (newOrderStatus == OrderStatus.Complete &&
                        newTaskStatus == TaskStatus.Completed &&
                        !string.IsNullOrWhiteSpace(pendingSpecimen.CprsOrderNumber))
                    {
                        dtOrderedTestForVista = DAL.OrderedTest.GetOrderedTestForVista(pendingSpecimen.CprsOrderNumber);

                        if (dtOrderedTestForVista.Rows.Count != 0)
                        {
                            dtOrderedTestForVista.Rows[0][TABLE.OrderedTest.TaskStatusCode] = Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed).ToString();
                            dtOrderedTestForVista.Rows[0][TABLE.OrderedTest.OrderStatusCode] = Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Complete).ToString();
                        }
                    }
                }
            }
        }

        private static void GetTaskStatusForAccepting(IPendingSpecimenTestModel pendingSpecimen, IEnumerable<IPendingSpecimenTestModel> pendingTestList, out TaskStatus taskStatus, out OrderStatus orderStatus)
        {
            taskStatus = TaskStatus.Completed;
            orderStatus = OrderStatus.Complete;

            // If this is a TAS test we need to check if the associated test (AboRh or ABS) is
            // also being accepted at the same time.            
            if (pendingSpecimen.OrderableTest.OrderableTestId == OrderableTest.TAS)
            {
                taskStatus = Utility.GetTaskStatusCodeFromString(pendingSpecimen.OrderedTest.TaskStatusCode.ToString());
                orderStatus = pendingSpecimen.OrderedTest.OrderStatusCode;

                // Get PendingSpecimenTest data from the database
                var dsAllPendingSpecimenTests = DAL.PendingSpecimenTest.GetPendingSpecimenTests(pendingSpecimen.OrderedTest.Patient.PatientGuid, pendingSpecimen.PatientSpecimenGuid, pendingSpecimen.DivisionCode, PendingTestStatus.Pending, OrderableTest.TAS);
                // Convert DataTable to list of PendingSpecimenTestDao
                var pendingSpecimenDaoList = dsAllPendingSpecimenTests.Tables[0].ToList<PendingSpecimenTestDao>();

                //Filter out the current test we are evaluating
                var pendingResultIdList = pendingSpecimen.TestResultList.Select(s => s.PendingTestId);
                var associatedSpecimenDaoList = pendingSpecimenDaoList.Where(x => !pendingResultIdList.Contains(x.PendingSpecimenTestId) && x.OrderedTestGuid == pendingSpecimen.OrderedTest.OrderedTestGuid).ToList();

                if (associatedSpecimenDaoList.Any())
                {
                    var associatedIdList = associatedSpecimenDaoList.Select(x => x.PendingSpecimenTestId);
                    var pendingTestListIdList = pendingTestList.SelectMany(x => x.TestResultList.Select(s => s.PendingTestId));
                    var associatedTestIsBeingAccepted = pendingTestListIdList.Any(x => associatedIdList.Contains(x));

                    if (associatedTestIsBeingAccepted)
                    {
                        taskStatus = TaskStatus.Completed;
                        orderStatus = OrderStatus.Complete;
                    }
                    else
                    {
                        taskStatus = TaskStatus.PendingReview;
                    }
                }
                else
                {
                    DataTable dtSpecimenTests = DAL.OrderedTest.GetTestResultsForTask(pendingSpecimen.OrderedTest.OrderedTestGuid);
                    var pendingBloodTestTypeList = pendingSpecimen.TestResultList.Select(s => s.TestType);
                    var foundAssociatedTest = false;

                    foreach (DataRow drSpecimenTest in dtSpecimenTests.Rows)
                    {
                        var specimentBloodTestType = (TestType)drSpecimenTest[TABLE.SpecimenTest.BloodTestTypeId];
                        foundAssociatedTest = !pendingBloodTestTypeList.Contains(specimentBloodTestType);

                        if (foundAssociatedTest)
                        {
                            // We found our match so break out
                            break;
                        }
                    }

                    if (foundAssociatedTest)
                    {
                        // Both TAS test should be be completed
                        taskStatus = TaskStatus.Completed;
                        orderStatus = OrderStatus.Complete;
                    }
                    else
                    {
                        taskStatus = TaskStatus.PartiallyCompleted;
                    }
                }
            }
        }

        private static void GetTaskStatusForTasRejecting(IPendingSpecimenTestModel pendingSpecimen, IEnumerable<IPendingSpecimenTestModel> pendingTestList, out TaskStatus taskStatus, out OrderStatus orderStatus, ref bool revertToPreviousStatus)
        {
            taskStatus = Utility.GetTaskStatusCodeFromString(pendingSpecimen.OrderedTest.TaskStatusCode.ToString());
            orderStatus = pendingSpecimen.OrderedTest.OrderStatusCode;
            revertToPreviousStatus = false;

            // If this is a TAS test we need to check if the associated test (AboRh or ABS) is
            // also being rejected at the same time.            
            if (pendingSpecimen.OrderableTest.OrderableTestId == OrderableTest.TAS)
            {
                // Get PendingSpecimenTest data from the database
                var dsAllPendingSpecimenTests = DAL.PendingSpecimenTest.GetPendingSpecimenTests(pendingSpecimen.OrderedTest.Patient.PatientGuid, pendingSpecimen.PatientSpecimenGuid, pendingSpecimen.DivisionCode, PendingTestStatus.Pending, OrderableTest.TAS);
                // Convert DataTable to list of PendingSpecimenTestDao
                var pendingSpecimenDaoList = dsAllPendingSpecimenTests.Tables[0].ToList<PendingSpecimenTestDao>();

                //Filter out the current test we are evaluating
                var pendingResultIdList = pendingSpecimen.TestResultList.Select(s => s.PendingTestId);
                var associatedSpecimenDaoList = pendingSpecimenDaoList.Where(x => !pendingResultIdList.Contains(x.PendingSpecimenTestId) && x.OrderedTestGuid == pendingSpecimen.OrderedTest.OrderedTestGuid).ToList();

                if (associatedSpecimenDaoList.Any())
                {
                    var associatedIdList = associatedSpecimenDaoList.Select(x => x.PendingSpecimenTestId);
                    var pendingTestListIdList = pendingTestList.SelectMany(x => x.TestResultList.Select(s => s.PendingTestId));
                    var associatedTestIsBeingRejected = pendingTestListIdList.Any(x => associatedIdList.Contains(x));

                    if (associatedTestIsBeingRejected)
                    {
                        revertToPreviousStatus = true;
                    }
                    else
                    {
                        taskStatus = TaskStatus.PendingReview;
                    }
                }
                else
                {
                    DataTable dtSpecimenTests = DAL.OrderedTest.GetTestResultsForTask(pendingSpecimen.OrderedTest.OrderedTestGuid);
                    var pendingBloodTestTypeList = pendingSpecimen.TestResultList.Select(s => s.TestType);
                    var foundAssociatedTest = false;

                    foreach (DataRow drSpecimenTest in dtSpecimenTests.Rows)
                    {
                        var specimentBloodTestType = (TestType)drSpecimenTest[TABLE.SpecimenTest.BloodTestTypeId];
                        foundAssociatedTest = !pendingBloodTestTypeList.Contains(specimentBloodTestType);

                        if (foundAssociatedTest)
                        {
                            // We found our match so break out
                            break;
                        }
                    }

                    if (foundAssociatedTest)
                    {
                        taskStatus = TaskStatus.PartiallyCompleted;
                    }
                    else
                    {
                        revertToPreviousStatus = true;
                    }
                }
            }
        }

        private static DataTable GetEmptyPendingSpecimenTestTableSchema()
        {
            DataTable dtPendingSpecimenTest = new DataTable(TABLE.PendingSpecimenTest.TableName);
            dtPendingSpecimenTest.Columns.Add(TABLE.PendingSpecimenTest.PendingSpecimenTestId, typeof(int));
            dtPendingSpecimenTest.Columns.Add(TABLE.PendingSpecimenTest.SpecimenTestGuid, typeof(Guid));
            dtPendingSpecimenTest.Columns.Add(TABLE.PendingSpecimenTest.PendingTestStatusId, typeof(byte));
            dtPendingSpecimenTest.Columns.Add(TABLE.PendingSpecimenTest.RejectionComment, typeof(string));
            dtPendingSpecimenTest.Columns.Add(TABLE.PendingSpecimenTest.LastUpdateUser, typeof(string));
            dtPendingSpecimenTest.Columns.Add(TABLE.PendingSpecimenTest.RowVersion, typeof(byte[]));
            dtPendingSpecimenTest.Columns.Add(TABLE.PendingSpecimenTest.LastUpdateFunctionId, typeof(int));
            return dtPendingSpecimenTest;
        }

        private static DataTable GetEmptyOrderedTestTableSchema()
        {
            DataTable dt = new DataTable(TABLE.OrderedTest.TableName);
            dt.Columns.Add(TABLE.OrderedTest.OrderedTestGuid, typeof(Guid));
            dt.Columns.Add(TABLE.OrderedTest.LastUpdateUser, typeof(string));
            dt.Columns.Add(TABLE.OrderedTest.RowVersion, typeof(byte[]));
            dt.Columns.Add(TABLE.OrderedTest.LastUpdateFunctionId, typeof(int));
            return dt;
        }

        private static DataTable GetEmptyOrderedTestTableSchemaForStatusUpdate()
        {
            var dtOrderedTest = new DataTable(TABLE.OrderedTest.TableName);
            dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestGuid, typeof(Guid));
            dtOrderedTest.Columns.Add(TABLE.OrderedTest.TaskStatusCode, typeof(string));
            dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderStatusCode, typeof(string));
            dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateUser, typeof(string));
            dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateFunctionId, typeof(int));
            dtOrderedTest.Columns.Add(TABLE.OrderedTest.RowVersion, typeof(byte[]));
            return dtOrderedTest;
        }
    }
}



